Attribute VB_Name = "Template"
'    This is a part of the source code for Pro/DESKTOP.
'    Copyright (C) 1999-2001 Parametric Technology Corporation.
'    All rights reserved.

Option Explicit

Private app As ProDESKTOP

Private ItCls As ItClass
Private MatrixCls As MatrixClass
Private VectorCls As VectorClass
Private BoxCls As BoxClass

Private ext As IProDExtensibility
Private uc1 As userCommand
Private uc2 As userCommand
Private uc3 As userCommand

Private Type DrawingScale
    Paper As Integer
    Model As Integer
End Type

Public Sub OnStartUp()
    Set app = GetApp
    Set ext = app
    
    ext.ImportVBComponent app.GetAppExecutablePath + "\..\Documentation\Addins\Macros\Template\TemplateDlg.frm"
    ext.ImportVBComponent app.GetAppExecutablePath + "\..\Documentation\Addins\Macros\Template\ViewDef.cls"

    Set uc1 = ext.AddUserCommand(barDrawing, menuDrawingTools, -1, GetResourceString(59), "Template", "Template.NewDrawing")
    Set uc2 = ext.AddUserCommand(barDesign, menuDesignTools, -1, GetResourceString(59), "Template", "Template.NewDrawing")
    Set uc3 = ext.AddUserCommand(barMain, menuMainTools, -1, GetResourceString(59), "Template", "Template.NewDrawing")

    uc1.SetPrompt GetResourceString(59)
    uc2.SetPrompt GetResourceString(59)
    uc3.SetPrompt GetResourceString(59)
End Sub

Public Sub OnCloseDown()
    Set app = GetApp
    Set ext = app
    
    Set uc1 = ext.GetUserCommand(barDrawing, menuDrawingTools, GetResourceString(59))
    Set uc2 = ext.GetUserCommand(barDesign, menuDesignTools, GetResourceString(59))
    Set uc3 = ext.GetUserCommand(barMain, menuMainTools, GetResourceString(59))
    
    ext.RemoveUserCommand uc1
    ext.RemoveUserCommand uc2
    ext.RemoveUserCommand uc3
    
    ext.RemoveVBComponent app.GetAppExecutablePath + "\..\Documentation\Addins\Macros\Template\TemplateDlg.frm"
    ext.RemoveVBComponent app.GetAppExecutablePath + "\..\Documentation\Addins\Macros\Template\ViewDef.cls"
End Sub

Public Sub NewDrawing()
    On Error GoTo errorHandler
    Dim dlg As New TemplateDlg
    dlg.Show
    On Error GoTo 0
Exit Sub
errorHandler:
    MsgBox GetResourceString(96)
End Sub

Public Sub ExecuteTemplate(templateName As String, design As aDesign)
    Set app = GetApp
    
    app.SetVisible True ' if run from VB

    ' setup common classes
    Set ItCls = app.GetClass("It")
    Set MatrixCls = app.GetClass("Matrix")
    Set VectorCls = app.GetClass("Vector")
    Set BoxCls = app.GetClass("Box")
    
    Dim api As helm
    Set api = app.TakeHelm

    Dim targetDrawingDoc As DrawingDocument, targetDrawing As aDrawing, targetSheet As aSheet
    
    Dim file As aFile
    Set file = app.LoadFile(templateName, "Drawing")
    
    Dim templateDrawing As aDrawing
    Set templateDrawing = file.GetHead()
    
    Dim sheetSet As ObjectSet
    Set sheetSet = templateDrawing.GetSheets
    
    If sheetSet.GetCount <> 1 Then GoTo Failed 'we expect to be called with a single sheet drawing
    
    Dim templateSheet As aSheet
    Set templateSheet = sheetSet.GetAnyMember
        
    Set targetDrawingDoc = app.NewDrawing
    Set targetDrawing = targetDrawingDoc.GetDrawing
    Set targetSheet = targetDrawingDoc.GetActiveSheet

    targetSheet.SetFormat templateSheet ' hack to keep 'templateDwg' from being garbage collected

    Dim callouts As ObjectSet
    Set callouts = GetAllNoteCallouts(templateSheet)

    CopySketches templateSheet, targetSheet
    CreateNotes callouts, targetSheet, targetDrawing

    Dim views As New Collection
    GetViews views, templateSheet

    If design Is Nothing And views.Count > 0 Then
        Set design = app.NewPart.GetDesign
    End If

    If views.Count > 0 Then
        If Not SetViewInfo(views, callouts, design) Then Exit Sub
        
        Dim dwgScale As DrawingScale
        GetGlobalViewScale views, design, dwgScale

        targetSheet.SetViewScale dwgScale.Paper, dwgScale.Model
    End If

    If views.Count > 0 Then
        CreateViews targetSheet, design, views
    End If

    Dim formatSheet As aSheet
    Set formatSheet = templateSheet.GetFormat

    If formatSheet Is Nothing Then    ' set the size from the template drawing
        targetSheet.SetSize templateSheet.GetWidth, templateSheet.GetHeight
    Else
        targetSheet.SetFormat formatSheet
        targetSheet.SetSize formatSheet.GetWidth, formatSheet.GetHeight
    End If
    
    app.ActivateDoc targetDrawingDoc
    Set targetDrawingDoc = Nothing
    
    Set file = Nothing
    
    api.CommitCalls "New Drawing", False
Exit Sub
Failed:
    MsgBox ("Template drawing can't contain more than one sheet")
End Sub

Private Function GetAllNoteCallouts(sheet As aSheet) As ObjectSet
    Dim callouts As ObjectSet
    Set callouts = app.GetClass("ObjectSet").CreateAObjectSet()
    
    Dim groups As ObjectSet
    Set groups = sheet.GetCalloutGroups

    Dim groupIt As iterator
    Set groupIt = ItCls.CreateAObjectIt(groups)
    
    groupIt.start
    Do While groupIt.IsActive
        Dim group As aCalloutGroup
        Set group = groupIt.Current

        Dim calloutSet As ObjectSet
        Set calloutSet = group.GetContents

        Dim calloutIt As iterator
        Set calloutIt = ItCls.CreateAObjectIt(calloutSet)

        calloutIt.start
        Do While calloutIt.IsActive
            Dim callout As aCallout
            Set callout = calloutIt.Current
            
            If callout.IsA("NoteCallout") Then
                callouts.AddMember callout
            End If
            calloutIt.Next
        Loop
        
        groupIt.Next
    Loop
    
    Set GetAllNoteCallouts = callouts
End Function

Private Sub AddView(views As Collection, viewbox As zBox)
    Dim v As New ViewDef
    Set v.Viewport = viewbox
    Set v.Orientation = Nothing
    Set v.Workplane = Nothing
    v.ScaleFactor = 0
    v.ShowHidden = True
    v.ShowSmooth = True
    v.LinkToWorkplane = False
    views.Add v
End Sub

Private Sub GetViews(views As Collection, templateSheet As aSheet)
    Dim Sketches As ObjectSet
    Set Sketches = templateSheet.GetWorkplane.GetSketches
    
    Dim sketchIt As iterator
    Set sketchIt = ItCls.CreateAObjectIt(Sketches)

    sketchIt.start
    Do While sketchIt.IsActive
        Dim sketch As aSketch
        Set sketch = sketchIt.Current
        
        ' sketch must be named "view..."
        If InStr(1, sketch.GetName, "view", vbTextCompare) = 1 Then
            Dim lines As ObjectSet
            Set lines = sketch.GetLines(False, True)
            
            Dim viewbox As zBox
            Dim nLines As Integer
            
            nLines = 0
    
            Dim lineIt As iterator
            Set lineIt = ItCls.CreateAObjectIt(lines)
            
            lineIt.start
            Do While lineIt.IsActive
                Dim line As aLine
                Set line = lineIt.Current
    
                Dim straight As zStraight
                Set straight = line.GetGeometry
                            
                Dim box As zBox
                Set box = BoxCls.CreateBoxAtPoint(straight.GetStart)
                If nLines = 0 Then
                    Set viewbox = box
                Else
                    Set viewbox = viewbox.unite(box)
                End If
                
                Set box = BoxCls.CreateBoxAtPoint(straight.GetEnd)
                Set viewbox = viewbox.unite(box)

                nLines = nLines + 1

                lineIt.Next
            Loop
            
            If Not nLines = 4 Then
                MsgBox GetResourceString(253) & Chr(13) & GetResourceString(254)
            Else
                AddView views, viewbox
            End If
        End If
        
        sketchIt.Next
    Loop
End Sub

Private Function SetViewInfo(views As Collection, callouts As ObjectSet, design As aDesign) As Boolean
    SetViewInfo = True
    Dim workplanes As ObjectSet

    Dim calloutIt As iterator
    Set calloutIt = ItCls.CreateAObjectIt(callouts)

    Dim v As ViewDef
    For Each v In views
        Dim min As zVector, max As zVector
        Set min = v.Viewport.GetMin
        Set max = v.Viewport.GetMax

        calloutIt.start
        Do While calloutIt.IsActive
            Dim callout As aNoteCallout, group As aCalloutGroup
            Set callout = calloutIt.Current
            Set group = callout.GetParent("CalloutGroup")
            
            Dim pos As zVector
            Set pos = group.GetControlPoint(topCenter)
            Dim x As Double, Y As Double
            x = pos.GetAt(0)
            Y = pos.GetAt(1)

            If x > min.GetAt(0) And Y > min.GetAt(1) And x < max.GetAt(0) And Y < max.GetAt(1) Then
                Dim text As String
                text = callout.GetNote.GetText
                'Debug.Print "view text: " & text
                
                If InStr(1, text, "orient ", vbTextCompare) = 1 Then
                    text = Mid(text, 8)
                    If InStr(1, text, "workplane ", vbTextCompare) = 1 Then
                        text = Mid(text, 11)
                        If design Is Nothing Then
                            MsgBox GetResourceString(255) & Chr(13) & GetResourceString(256)
                        ElseIf workplanes Is Nothing Then
                            Set workplanes = design.GetWorkplanes
                        End If
                        Set v.Workplane = GetWorkplane(workplanes, text)
                        If v.Workplane Is Nothing Then
                            MsgBox MakeString(GetResourceString(257), text) & Chr(13) & GetResourceString(258)
                            Set v.Workplane = GetWorkplane(workplanes, "base")
                            If v.Workplane Is Nothing Then
                                Set v.Workplane = workplanes.GetAnyMember
                            End If
                        End If
                        Set v.Orientation = GetWorkplaneProjection(v.Workplane)
                    ElseIf InStr(1, text, "topleft", vbTextCompare) = 1 Then
                        Set v.Orientation = MatrixCls.CreateMatrix(0, 0, 1, 0, 0, 1, 0, 0, -1, 0, 0, 0, 0, 0, 0, 1)
                    ElseIf InStr(1, text, "topright", vbTextCompare) = 1 Then
                        Set v.Orientation = MatrixCls.CreateMatrix(0, 0, -1, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1)
                    ElseIf InStr(1, text, "topback", vbTextCompare) = 1 Then
                        Set v.Orientation = MatrixCls.CreateMatrix(1, 0, 0, 0, 0, 0, -1, 0, 0, 1, 0, 0, 0, 0, 0, 1)
                    ElseIf InStr(1, text, "top", vbTextCompare) = 1 Then
                        Set v.Orientation = MatrixCls.CreateMatrix(1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1)
                    ElseIf InStr(1, text, "plan", vbTextCompare) = 1 Then
                        Set v.Orientation = MatrixCls.CreateMatrix(1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1)
                    ElseIf InStr(1, text, "bottom", vbTextCompare) = 1 Then
                        Set v.Orientation = MatrixCls.CreateMatrix(1, 0, 0, 0, 0, -1, 0, 0, 0, 0, -1, 0, 0, 0, 0, 1)
                    ElseIf InStr(1, text, "front", vbTextCompare) = 1 Then
                        Set v.Orientation = MatrixCls.CreateMatrix(1, 0, 0, 0, 0, 0, 1, 0, 0, -1, 0, 0, 0, 0, 0, 1)
                    ElseIf InStr(1, text, "back", vbTextCompare) = 1 Then
                        Set v.Orientation = MatrixCls.CreateMatrix(-1, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 1)
                    ElseIf InStr(1, text, "left", vbTextCompare) = 1 Then
                        Set v.Orientation = MatrixCls.CreateMatrix(0, -1, 0, 0, 0, 0, 1, 0, -1, 0, 0, 0, 0, 0, 0, 1)
                    ElseIf InStr(1, text, "right", vbTextCompare) = 1 Then
                        Set v.Orientation = MatrixCls.CreateMatrix(0, 1, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 1)
                    ElseIf InStr(1, text, "isometric", vbTextCompare) = 1 Then
                        Set v.Orientation = GetIsometricProjection()
                    ElseIf InStr(1, text, "trimetric", vbTextCompare) = 1 Then
                        Set v.Orientation = GetTrimetricProjection()
                    Else
                        MsgBox MakeString(GetResourceString(259), text)
                    End If
                ElseIf InStr(1, text, "hidden ", vbTextCompare) = 1 Then
                    text = Mid(text, 8)
                    If InStr(1, text, "on", vbTextCompare) = 1 Then
                        v.ShowHidden = True
                    ElseIf InStr(1, text, "off", vbTextCompare) = 1 Then
                        v.ShowHidden = False
                    Else
                        MsgBox MakeString(GetResourceString(260), text)
                    End If
                ElseIf InStr(1, text, "smooth ", vbTextCompare) = 1 Then
                    text = Mid(text, 8)
                    If InStr(1, text, "on", vbTextCompare) = 1 Then
                        v.ShowSmooth = True
                    ElseIf InStr(1, text, "off", vbTextCompare) = 1 Then
                        v.ShowSmooth = False
                    Else
                        MsgBox MakeString(GetResourceString(261), text)
                    End If
                ElseIf InStr(1, text, "scale ", vbTextCompare) = 1 Then
                    text = Mid(text, 7)
                    If InStr(1, text, "fit", vbTextCompare) = 1 Then
                        v.ScaleFactor = -1
                    Else
                        On Error GoTo badValue
                        v.ScaleFactor = CDbl(text)  ' FIXME: validate
                        On Error GoTo 0
                        If v.ScaleFactor <= 0 Then GoTo badValue
                    End If
                ElseIf InStr(1, text, "linkToWorkplane", vbTextCompare) = 1 Then
                    v.LinkToWorkplane = True
                ElseIf Not Mid(text, 1, 1) = "$" Then
                    MsgBox MakeString(GetResourceString(262), text)
                End If
            End If

            calloutIt.Next
        Loop    ' calloutIt

        If v.Orientation Is Nothing Then
            Set v.Orientation = MatrixCls.CreateMatrix(1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1)
        End If
    Next v
Exit Function

badValue:
    MsgBox GetResourceString(570)
    SetViewInfo = False
End Function

Private Function GetBoundingBoxOfDesign(design As aDesign, transform As zMatrix) As zBox
    ' more intelligent aDesign.GetBoundingBox taking account of what a view will display
    ' if there are no solids in the design
    Dim designBox As zBox
    Set designBox = design.GetBoundingBox(transform)
    
    If designBox.IsEmpty() Then
        Dim workplanes As ObjectSet
        Set workplanes = design.GetWorkplanes
        Dim it As iterator
        Set it = ItCls.CreateAObjectIt(workplanes)
        it.start
        Do While it.IsActive
            Dim wp As aWorkplane
            Set wp = it.Current

            Dim wpBox As zBox
            Set wpBox = Nothing

            Dim Sketches As ObjectSet
            Set Sketches = wp.GetSketches
            
            Dim skIt As iterator
            Set skIt = ItCls.CreateAObjectIt(Sketches)
            
            skIt.start
            Do While skIt.IsActive
                Dim sk As aSketch
                Set sk = skIt.Current
                If sk.IsVisible Then
                    Dim box As zBox
                    Set box = sk.GetBoundingBox(transform)
                    If Not box.IsEmpty Then
                        If wpBox Is Nothing Then
                            Set wpBox = box
                        Else
                            Set wpBox = wpBox.unite(box)
                        End If
                    End If
                End If
                skIt.Next
            Loop
            
            If Not wpBox Is Nothing Then
                Set designBox = designBox.unite(wpBox)
            End If

            it.Next
        Loop
    End If
    
    If designBox.IsEmpty() Then ' a completely empty design will create a view with a bounding box 100mm cube
        Set designBox = BoxCls.CreateBox(VectorCls.CreateVector(-0.05, -0.05, -0.05), VectorCls.CreateVector(0.05, 0.05, 0.05))
        designBox.transform transform
    End If
    
    Set GetBoundingBoxOfDesign = designBox
End Function

Private Sub GetGlobalViewScale(views As Collection, design As aDesign, dwgScale As DrawingScale)
    Dim overallScale As Double
    overallScale = -1

    Dim v As ViewDef
    For Each v In views
        Dim designBox As zBox
        Set designBox = GetBoundingBoxOfDesign(design, v.Orientation)

        Dim sca As Double
        If designBox.GetWidth = 0 Then
            sca = v.Viewport.GetHeight / designBox.GetHeight
        ElseIf designBox.GetHeight = 0 Then
            sca = v.Viewport.GetWidth / designBox.GetWidth
        Else
            Dim xSca As Double, ySca As Double
            xSca = v.Viewport.GetWidth / designBox.GetWidth
            ySca = v.Viewport.GetHeight / designBox.GetHeight
            If xSca < ySca Then
                sca = xSca
            Else
                sca = ySca
            End If
        End If

        If v.ScaleFactor = -1 Then  ' fit to viewport
            v.ScaleFactor = sca
        ElseIf v.ScaleFactor = 0 Then   ' calculate overall
            If overallScale = -1 Then
                overallScale = sca
            ElseIf sca < overallScale Then
                overallScale = sca
            End If
        End If
    Next v

    If overallScale = -1 Then
        dwgScale.Paper = 1
        dwgScale.Model = 1
    Else
        NormalizeScale overallScale, dwgScale
        overallScale = dwgScale.Paper / dwgScale.Model
        For Each v In views
            If v.ScaleFactor = 0 Then
                v.ScaleFactor = overallScale
            End If
        Next v
    End If
End Sub

Private Sub NormalizeScale(s As Double, dwgScale As DrawingScale)
    Dim roundedLog As Integer
    roundedLog = CInt(Log(s) / Log(10#) + 0.5)
    
    Dim f As Double
    f = s / (10 ^ roundedLog)

    ' we will probably need to tweak the various commonly used scale factors as well as the clamp boundaries
    If f < 0.125 Then    ' approx 0.1
        dwgScale.Paper = 1
        dwgScale.Model = 10
    ElseIf f < 0.18 Then    ' approx 0.15
        dwgScale.Paper = 3
        dwgScale.Model = 20
    ElseIf f < 0.24 Then    ' approx 0.2
        dwgScale.Paper = 1
        dwgScale.Model = 5
    ElseIf f < 0.28 Then    ' approx 0.25
        dwgScale.Paper = 1
        dwgScale.Model = 4
    ElseIf f < 0.32 Then    ' approx 0.3
        dwgScale.Paper = 3
        dwgScale.Model = 10
    ElseIf f < 0.35 Then    ' approx 0.333333
        dwgScale.Paper = 1
        dwgScale.Model = 3
    ElseIf f < 0.42 Then    ' approx 0.4
        dwgScale.Paper = 2
        dwgScale.Model = 5
    ElseIf f < 0.55 Then    ' approx 0.5
        dwgScale.Paper = 1
        dwgScale.Model = 2
    ElseIf f < 0.63 Then    ' approx 0.6
        dwgScale.Paper = 3
        dwgScale.Model = 5
    ElseIf f < 0.7 Then     ' approx 0.666666
        dwgScale.Paper = 2
        dwgScale.Model = 3
    ElseIf f < 0.77 Then    ' approx 0.75
        dwgScale.Paper = 3
        dwgScale.Model = 4
    ElseIf f < 0.85 Then    ' approx 0.8
        dwgScale.Paper = 4
        dwgScale.Model = 5
    Else
        dwgScale.Paper = 1
        dwgScale.Model = 1
    End If

    If roundedLog < 0 Then
        dwgScale.Model = dwgScale.Model * 10 ^ -roundedLog
    ElseIf roundedLog > 0 Then
        dwgScale.Paper = dwgScale.Paper * 10 ^ roundedLog
    End If
    
    Dim lower As Double, upper As Double
    If dwgScale.Paper = dwgScale.Model Then
        dwgScale.Paper = 1
        dwgScale.Model = 1
        GoTo finish
    ElseIf dwgScale.Paper > dwgScale.Model Then
        upper = dwgScale.Paper
        lower = dwgScale.Model
    Else
        upper = dwgScale.Model
        lower = dwgScale.Paper
    End If
        
    Dim i As Integer, d As Integer
    For i = 1 To lower
        If lower Mod i = 0 Then
            d = lower / i
            If upper Mod d = 0 Then
                dwgScale.Paper = dwgScale.Paper / d
                dwgScale.Model = dwgScale.Model / d
                GoTo finish
            End If
        End If
    Next

finish:
    'Debug.Print dwgScale.Paper, dwgScale.Model, dwgScale.Paper / dwgScale.Model
End Sub

Private Sub CreateViews(targetSheet As aSheet, design As aDesign, views As Collection)
    Dim viewNumber As Integer   ' used to name views
    viewNumber = 1

    Dim v As ViewDef
    For Each v In views
        Dim view As aView
        
        If v.LinkToWorkplane And Not v.Workplane Is Nothing Then
            Set view = app.GetClass("View").CreateWorkplaneView(targetSheet, design, v.Workplane)
        Else
            Set view = app.GetClass("View").CreateView(targetSheet, design)
        End If

        view.SetShowHidden v.ShowHidden
        view.SetShowSmooth v.ShowSmooth
        view.SetName GetResourceString(408) & viewNumber
                viewNumber = viewNumber + 1

        Dim designBox As zBox
        Set designBox = GetBoundingBoxOfDesign(design, v.Orientation)

        Dim adjustScale As zMatrix
        Set adjustScale = MatrixCls.CreateScaleMatrix(v.ScaleFactor)

        Dim designPos As zVector
        Set designPos = VectorCls.CreateVector(designBox.GetCenter.GetAt(0) * v.ScaleFactor, designBox.GetCenter.GetAt(1) * v.ScaleFactor, 0)

        Dim pos As zVector
        Set pos = v.Viewport.GetCenter.subtract(designPos)

        Dim center As zMatrix
        Set center = MatrixCls.CreateTranslationMatrix(pos)

        Dim trans As zMatrix
        Set trans = center.MultiplyByMatrix(adjustScale)

        If Not v.LinkToWorkplane Then
            Set trans = trans.MultiplyByMatrix(v.Orientation)
        End If

        view.transform trans
        view.UpdateImages
    Next v
End Sub

Private Function GetWorkplane(workplanes As ObjectSet, name As String) As aWorkplane
    Dim it As iterator
    Set it = ItCls.CreateAObjectIt(workplanes)
    it.start
    Do While it.IsActive
        Dim wp As aWorkplane
        Set wp = it.Current
        If wp.GetName = name Then
            Set GetWorkplane = wp
            Exit Function
        End If
        it.Next
    Loop
End Function

Private Function GetWorkplaneProjection(wp As aWorkplane) As zMatrix
    Dim hor As zDirection, ver As zDirection, eye As zVector
    Set hor = wp.GetLocalX
    Set ver = wp.GetLocalY
    Set eye = hor.Cross(ver)
    Set GetWorkplaneProjection = MatrixCls.CreateMatrix(hor.GetAt(0), hor.GetAt(1), hor.GetAt(2), 0, ver.GetAt(0), ver.GetAt(1), ver.GetAt(2), 0, eye.GetAt(0), eye.GetAt(1), eye.GetAt(2), 0, 0, 0, 0, 1)
End Function

Private Function GetIsometricProjection() As zMatrix
    Set GetIsometricProjection = MatrixCls.CreateMatrix(1 / Sqr(2), 1 / Sqr(2), 0, 0, -1 / Sqr(6), 1 / Sqr(6), Sqr(2# / 3), 0, 1 / Sqr(3), -1 / Sqr(3), 1 / Sqr(3), 0, 0, 0, 0, 1)
End Function

Private Function GetTrimetricProjection() As zMatrix
    Set GetTrimetricProjection = MatrixCls.CreateMatrix(Sqr(3) / 2, 0.5, 0, 0, -0.25, Sqr(3#) / 4, Sqr(3#) / 2, 0, Sqr(3#) / 4, -0.75, 0.5, 0, 0, 0, 0, 1)
End Function

Private Sub CopySketches(templateSheet As aSheet, targetSheet As aSheet)
    Dim wp As aWorkplane
    Set wp = targetSheet.GetWorkplane

    Dim Sketches As ObjectSet
    Set Sketches = templateSheet.GetWorkplane.GetSketches
    
    Dim sketchIt As iterator
    Set sketchIt = ItCls.CreateAObjectIt(Sketches)

    Dim sketchNumber As Integer ' used to name sketches
        sketchNumber = 1

    sketchIt.start
    Do While sketchIt.IsActive
        Dim sketch As aSketch
        Set sketch = sketchIt.Current
        
        ' sketch must be named "copy..."
        If InStr(1, sketch.GetName, "copy", vbTextCompare) = 1 Then
            Dim name As String
            name = GetResourceString(409) & sketchNumber
            sketchNumber = sketchNumber + 1

            Dim newSketch As aSketch
            Set newSketch = wp.CreateSketch(name)
            newSketch.SetColor sketch.GetColor
            newSketch.SetLineWidth sketch.GetLineWidth
            newSketch.SetFilled sketch.IsFilled
            newSketch.SetRigid sketch.IsRigid

            Dim lines As ObjectSet
            Set lines = sketch.GetLines(True, False)    ' don't copy construction lines

            Dim lineIt As iterator
            Set lineIt = ItCls.CreateAObjectIt(lines)
            
            lineIt.start
            Do While lineIt.IsActive
                Dim line As aLine, newLine As aLine
                Set line = lineIt.Current
                
                Dim curve As zCurve
                Set curve = line.GetGeometry
                Set newLine = newSketch.CreateLine(curve)
                
                newLine.SetConstruction line.IsConstruction
                lineIt.Next
            Loop
        End If
        
        sketchIt.Next
    Loop
End Sub

Private Sub CreateNotes(callouts As ObjectSet, targetSheet As aSheet, targetDwg As aDrawing)
    Dim calloutIt As iterator
    Set calloutIt = ItCls.CreateAObjectIt(callouts)

    calloutIt.start
    Do While calloutIt.IsActive
        Dim callout As aNoteCallout
        Set callout = calloutIt.Current
        
        Dim create As Boolean
        create = False
        
        Dim text As String, cp As String
        text = callout.GetNote.GetText
        If InStr(1, text, "$copy", vbTextCompare) = 1 Then
            cp = Mid(text, 6, 1)
            text = Trim(Mid(text, 7))
            create = True
        ElseIf InStr(1, text, "$input", vbTextCompare) = 1 Then
            cp = Mid(text, 7, 1)
            text = Trim(Mid(text, 8))
            text = InputBox(MakeString(GetResourceString(263), text))
            create = True
        ElseIf InStr(1, text, "$date", vbTextCompare) = 1 Then
            cp = Mid(text, 6, 1)
            text = Date
            create = True
        ElseIf InStr(1, text, "$time", vbTextCompare) = 1 Then
            cp = Mid(text, 6, 1)
            text = Time
            create = True
        End If

        If create Then
            Dim controlPoint As Long
            controlPoint = topLeft

            If cp = "l" Then
                controlPoint = topLeft
            ElseIf cp = "c" Then
                controlPoint = topCenter
            ElseIf cp = "r" Then
                controlPoint = topRight
            ElseIf Not cp = " " And Not cp = "" Then
                MsgBox GetResourceString(264)
            End If

            Dim group As aCalloutGroup
            Set group = callout.GetParent("CalloutGroup")

            Dim pos As zVector
            Set pos = group.GetControlPoint(controlPoint)

            Dim newNote As aNote, newCallout As aNoteCallout, newGroup As aCalloutGroup
            Set newNote = app.GetClass("Note").CreateNote(targetDwg, text)
            Set newCallout = app.GetClass("NoteCallout").CreateNoteCallout(newNote)
            Set newGroup = app.GetClass("CalloutGroup").CreateCalloutGroup(newCallout, pos, group.GetTextHeight)
            newGroup.SetControlPoint controlPoint, pos

            Dim font As String, bold As Boolean, italic As Boolean, outline As Boolean
            font = group.GetFontDescriptor(bold, italic, outline)
            newGroup.SetFontDescriptor font, bold, italic, outline
            newGroup.SetPen group.GetPen
            newGroup.SetTransparent group.IsTransparent
            targetSheet.AddCalloutGroup newGroup
        End If
        calloutIt.Next
    Loop
End Sub
